/*
 * C compiler
 * ==========
 *
 * Copyright 1989, 1990, 1991 Christoph van Wuellen.
 * Credits to Matthew Brandt.
 * All commercial rights reserved.
 *
 * This compiler may be redistributed as long there is no
 * commercial interest. The compiler must not be redistributed
 * without its full sources. This notice must stay intact.
 *
 * History:
 *
 * 1989   starting an 68000 C compiler, starting with material
 *        originally by M. Brandt
 * 1990   68000 C compiler further bug fixes
 *        started i386 port (December)
 * 1991   i386 port finished (January)
 *        further corrections in the front end and in the 68000
 *        code generator.
 *        The next port will be a SPARC port
 */

/*****************************************************************************/

#include "chdr.h"
#include "expr.h"
#include "cglbdec.h"
#include "proto.h"

/********************************************************* Macro Definitions */

/* macros for querying character type */
#define	is_digit(c)	(chtable[(c)] & A_DIGIT)
#define	is_idch(c)	(chtable[(c)] & A_IDCHAR)
#define	is_octal(c)	(chtable[(c)] & A_OCTAL)
#define	is_hex(c)	(chtable[(c)] & A_HEX)
#define	is_space(c)	(chtable[(c)] & A_SPACE)
#define	is_print(c)	(chtable[(c)] & A_PRINT)

#define	END_OF_BUFFER	((CHAR)'\0')

#define	END_STRING	9999	/* end of string encountered */
#define MAXKEY	((HASH) 631)	/* hash table size */
#define	BUFLEN		512	/* length of input buffer */
#define	ID_LENGTH	127	/* usual maximum length of identifier */
#define	NIL_WORD	((XWORD *)0)

/********************************************************** Type Definitions */

/* Attributes of input characters */
enum e_attribute {
    A_DIGIT = 1,		/* character is a decimal digit */
    A_IDCHAR = 2,		/* character is an identifier character */
    A_OCTAL = 4,		/* character is an octal digit */
    A_HEX = 8,			/* character is a hex digit */
    A_SPACE = 16,		/* character is a space character */
    A_PRINT = 32		/* character is printable */
};

typedef struct _word {
    const CHAR *spelling;	/* characters of the string */
    size_t  length;		/* length of the string */
    TOKEN   sy;			/* string type */
    struct _word *next;		/* next string in the hash chain */
} XWORD, *WPTR;
typedef enum e_attribute Attribute;
typedef unsigned HASH;		/* value generated by hash function */

/********************************************************** Static Variables */

static BOOL overflow;		/* indicator for constant overflow */
static WPTR hashtable[MAXKEY];	/* hash table */

/*
 *   The input buffer has several pointers which control its operation; the
 *   objective is to try to make it as efficient as possible to process
 *   individual characters.  The primary way that this is done is to
 *   reduce the number of times that it is necessary to check to see if
 *   the end of the currently read input has been reached.
 *
 *   bufstart        This points to the start of the input buffer.  If the
 *                   buffer is enlarged then this value could change.
 *
 *   bufend          This points to the byte BEYOND the end of the input
 *                   buffer.  It is used to make the physical end of the
 *                   input buffer - the end of the characters read almost
 *                   certainly occurs before this point.
 *
 *   buflimit        This points to the byte beyond the current point
 *                   where input has been read into the input buffer.  This
 *                   always points to a \0 byte.
 *
 *   symstart        This points to the start of the current symbol.
 *
 *   bufcur          The current position in the input buffer.
 *
 *           symstart   bufcur
 *           |          |
 *           |          |
 *          \|/        \|/
 *    +----------------------------------+
 *    |                       \0         |
 *    +----------------------------------+
 *   /|\          /|\        /|\        /|\
 *    |            |          |          |
 *    |            |          |          |
 *    bufstart     bufmove    buflimit   bufend
 */

static CHAR *bufstart = NIL_CHAR;	/* start of the input buffer */
static CHAR *bufmove = NIL_CHAR;	/* mid point of the input buffer */
static CHAR *bufend = NIL_CHAR;	/* end of the input buffer */
static CHAR *buflimit = NIL_CHAR;	/* input buffer filled to this point */
static size_t buflen = (size_t) 0;	/* length of the input buffer */
static CHAR *symstart = NIL_CHAR;	/* start of the current symbol */
static CHAR *bufcur = NIL_CHAR;	/* current character in the buffer */
static int end_of_file = FALSE;	/* flag to indicate end of file */
static CHAR *strstart = NIL_CHAR;	/* buffer for strings */
static size_t strbuflen = (size_t) 0;	/* length of the string buffer */

static Attribute table[257];	/* table to implement ctype(3) funcs */
static Attribute *chtable = &table[1];	/* efficient access to table */

static const CHAR *pp_pragma;	/* pre-processor keyword pragma */
static const CHAR *pp_ident;	/* pre-processor keyword ident */
static const CHAR *pp_file;	/* pre-processor keyword file */
static const CHAR *pp_line;	/* pre-processor keyword line */

/*********************************************** Static Function Definitions */

static const CHAR *found P_ ((const CHAR *, size_t, BOOL));
static const CHAR *insert P_ ((const CHAR *, size_t, TOKEN, unsigned));
static HASH hash P_ ((const CHAR *, size_t));
static int getsch P_ ((BOOL));
static int radix36 P_ ((CHAR));
static void getdecimal P_ ((void));
static void getexp P_ ((void));
static void getfrac P_ ((void));
static void gethex P_ ((void));
static void getid P_ ((void));
static void getnum P_ ((void));
static void getoctal P_ ((void));
static void nextch P_ ((void));
static void nextline P_ ((void));
static void preprocessor_directive P_ ((void));
static void skip_space P_ ((void));
static void test_int P_ ((BOOL));

/*****************************************************************************/

/*
 *  setattributes() - defines attributes for specified characters
 */
static void setattributes P2 (const CHAR *, str, Attribute, attr)
{
    for (; *str; str++) {
	chtable[(int) *str] = (Attribute) (chtable[(int) *str] | attr);
    }
}


/*
 *  hash() - calculate the key for the hash function
 */
static HASH hash P2 (const CHAR *, p, size_t, length)
{
    register HASH sum;

    for (sum = (HASH) 0; length; length--)
	sum += (sum << 5) + (HASH) (*p++);
    return (sum % MAXKEY);
}


/*
 *  insert() - insert a word into the hash table
 */
static const CHAR *insert P4 (const CHAR *, p, size_t, length, TOKEN, sy, HASH, keyno)
{
    WPTR    wp;
    CHAR   *s;
    const CHAR *end;

    wp = (WPTR) galloc (sizeof (XWORD) + length + (size_t) 1);
    wp->next = hashtable[keyno];
    hashtable[keyno] = wp;
    wp->sy = sy;
    wp->length = length;
    s = (CHAR *) wp + sizeof (XWORD);
    wp->spelling = (const CHAR *) s;
    for (end = p + length; p < end;) {
	*s++ = *p++;
    }
    *s = (CHAR) '\0';

    return wp->spelling;
}


/*
 *  quick_insert() - insert word into hash table.
 *
 *  Insert a word into the hash table ... but there is no need to
 *  put the symbol spelling into the spelling table.
 */
static const CHAR *quick_insert P3 (const CHAR *, p, size_t, length, TOKEN, sy)
{
    HASH    keyno = hash (p, length);
    WPTR    wp = (WPTR) galloc (sizeof (XWORD));

    wp->next = hashtable[keyno];
    hashtable[keyno] = wp;
    wp->sy = sy;
    wp->length = length;
    wp->spelling = p;
    return wp->spelling;
}

/*
 *  found() - lookup a keyword in the string table.
 *
 *  Check to see if the name is a keyword.
 *  If it is a keyword then set lastst to the token.
 *
 *  A pointer to the string in the spelling table for the given name is
 *  returned.  This means that the same name always points to the same
 *  place in the spelling table - thus to compare names in the rest of the
 *  compiler it is only necessary to compare pointers!
 */
static const CHAR *found P3 (const CHAR *, p, size_t, length, BOOL, is_identifier)
{
    register WPTR wp;
    HASH    keyno = hash (p, length);

    for (wp = hashtable[keyno]; wp; wp = wp->next) {
	register size_t len = wp->length;

	if (length == len) {
	    register const CHAR *s1 = p;
	    register const CHAR *s2 = wp->spelling;

	    while (*s1++ == *s2++) {
		len--;
		if (len == (size_t) 0) {
		    lastst = wp->sy;
#ifdef FACIST
		    if (is_identifier && lastst == kw_id) {
			message (WARN_KEYWORD, wp->spelling);
			lastst = tk_id;
		    }
#endif /* FACIST */
		    return (wp->spelling);
		}
	    }
	}
    }
    return insert (p, length, lastst = tk_id, keyno);
}

/*
 *  nextline() - read next buffer from source file
 *
 *  An end-of-buffer character has been encountered in the input buffer.
 *  This could either be a genuine end-of-buffer character or else just
 *  a zero character in the current input buffer.
 *
 *  A dummy zero character is always placed one character beyond
 *  the characters read into the input buffer.  This allows the
 *  check for the end of the input buffer be performed only when
 *  this character occurs and not after every character.
 */
static void nextline P0 (void)
{
    if (bufcur >= buflimit) {
	/*
	 *   End of input currently read encountered
	 */
	size_t  len;

	if (symstart >= bufmove) {
	    /*
	     *  The start of the symbol occurs near the end of the input
	     *  buffer.   Move the symbol to the start of the input
	     *  buffer.
	     */
	    CHAR   *p1, *p2;

	    for (p1 = bufstart, p2 = symstart; p2 <= bufcur;) {
		*p1++ = *p2++;
	    }
	    symstart = bufstart;
	    bufcur = p1 - 1;
	}
	/*
	 *  The start of the symbol is near the start of the input
	 *  buffer.   The symbol is therefore potentially a long
	 *  symbol so extend the size of the buffer.
	 */
	if (((int) (bufend - bufcur)) < BUFLEN) {
	    size_t  len2 = (size_t) (symstart - bufstart);

	    len = (size_t) (bufcur - bufstart) - (size_t) 1;
	    buflen += (size_t) BUFLEN;
	    bufstart = (CHAR *) realloc (bufstart, buflen + (size_t) 1);
	    bufcur = bufstart + len;
	    symstart = bufstart + len2;
	}
	len = fread ((void *) bufcur, (size_t) 1, (size_t) BUFLEN, input);
	if (len == (size_t) 0) {
	    end_of_file = TRUE;
	    *bufcur = END_OF_BUFFER;
	    return;
	}
	bufend = bufstart + buflen;
	bufmove = bufstart + ID_LENGTH;
	buflimit = bufcur + len;
	*buflimit = END_OF_BUFFER;
    }
}


/*
 *   initsym() - Initialize the scanner tables.
 */
void initsym P0 (void)
{
    static XWORD keywords[] =
    {
	{(const CHAR *) "auto", (size_t) 4, kw_auto, NIL_WORD},
	{(const CHAR *) "break", (size_t) 5, kw_break, NIL_WORD},
	{(const CHAR *) "case", (size_t) 4, kw_case, NIL_WORD},
	{(const CHAR *) "char", (size_t) 4, kw_char, NIL_WORD},
	{(const CHAR *) "continue", (size_t) 8, kw_continue, NIL_WORD},
	{(const CHAR *) "default", (size_t) 7, kw_default, NIL_WORD},
	{(const CHAR *) "do", (size_t) 2, kw_do, NIL_WORD},
	{(const CHAR *) "else", (size_t) 4, kw_else, NIL_WORD},
	{(const CHAR *) "enum", (size_t) 4, kw_enum, NIL_WORD},
	{(const CHAR *) "extern", (size_t) 6, kw_extern, NIL_WORD},
	{(const CHAR *) "for", (size_t) 3, kw_for, NIL_WORD},
	{(const CHAR *) "goto", (size_t) 4, kw_goto, NIL_WORD},
	{(const CHAR *) "if", (size_t) 2, kw_if, NIL_WORD},
	{(const CHAR *) "int", (size_t) 3, kw_int, NIL_WORD},
	{(const CHAR *) "long", (size_t) 4, kw_long, NIL_WORD},
	{(const CHAR *) "register", (size_t) 8, kw_register, NIL_WORD},
	{(const CHAR *) "return", (size_t) 6, kw_return, NIL_WORD},
	{(const CHAR *) "short", (size_t) 5, kw_short, NIL_WORD},
	{(const CHAR *) "sizeof", (size_t) 6, kw_sizeof, NIL_WORD},
	{(const CHAR *) "static", (size_t) 6, kw_static, NIL_WORD},
	{(const CHAR *) "struct", (size_t) 6, kw_struct, NIL_WORD},
	{(const CHAR *) "switch", (size_t) 6, kw_switch, NIL_WORD},
	{(const CHAR *) "typedef", (size_t) 7, kw_typedef, NIL_WORD},
	{(const CHAR *) "union", (size_t) 5, kw_union, NIL_WORD},
	{(const CHAR *) "unsigned", (size_t) 8, kw_unsigned, NIL_WORD},
	{(const CHAR *) "void", (size_t) 4, kw_void, NIL_WORD},
	{(const CHAR *) "while", (size_t) 5, kw_while, NIL_WORD},
#ifdef FLOAT_KEYWORDS
	{(const CHAR *) "double", (size_t) 6, kw_double, NIL_WORD},
	{(const CHAR *) "float", (size_t) 5, kw_float, NIL_WORD},
#endif				/* FLOAT_KEYWORDS */
#ifdef FACIST
    /* C++ keywords and operators */
	{(const CHAR *) "bool", (size_t) 4, kw_id, NIL_WORD},
	{(const CHAR *) "catch", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "class", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "const_cast", (size_t) 10, kw_id, NIL_WORD},
	{(const CHAR *) "delete", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "dynamic_cast", (size_t) 12, kw_id, NIL_WORD},
	{(const CHAR *) "false", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "friend", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "inline", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "mutable", (size_t) 7, kw_id, NIL_WORD},
	{(const CHAR *) "namespace", (size_t) 9, kw_id, NIL_WORD},
	{(const CHAR *) "new", (size_t) 3, kw_id, NIL_WORD},
	{(const CHAR *) "operator", (size_t) 8, kw_id, NIL_WORD},
	{(const CHAR *) "private", (size_t) 7, kw_id, NIL_WORD},
	{(const CHAR *) "public", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "reinterpret_cast", (size_t) 16, kw_id, NIL_WORD},
	{(const CHAR *) "static_cast", (size_t) 11, kw_id, NIL_WORD},
	{(const CHAR *) "template", (size_t) 8, kw_id, NIL_WORD},
	{(const CHAR *) "this", (size_t) 4, kw_id, NIL_WORD},
	{(const CHAR *) "throw", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "true", (size_t) 4, kw_id, NIL_WORD},
	{(const CHAR *) "try", (size_t) 3, kw_id, NIL_WORD},
	{(const CHAR *) "typeid", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "using", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "virtual", (size_t) 7, kw_id, NIL_WORD},

	{(const CHAR *) "bitand", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "and", (size_t) 3, kw_id, NIL_WORD},
	{(const CHAR *) "bitor", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "or", (size_t) 2, kw_id, NIL_WORD},
	{(const CHAR *) "xor", (size_t) 3, kw_id, NIL_WORD},
	{(const CHAR *) "compl", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "and_eq", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "or_eq", (size_t) 5, kw_id, NIL_WORD},
	{(const CHAR *) "xor_eq", (size_t) 6, kw_id, NIL_WORD},
	{(const CHAR *) "not", (size_t) 3, kw_id, NIL_WORD},
	{(const CHAR *) "not_eq", (size_t) 6, kw_id, NIL_WORD},
#endif				/* FACIST */
	{NIL_CHAR, (size_t) 0, (TOKEN) 0, NIL_WORD}
    };
    static XWORD ansi_keywords[] =
    {
	{(const CHAR *) "const", (size_t) 5, kw_const, NIL_WORD},
	{(const CHAR *) "signed", (size_t) 6, kw_signed, NIL_WORD},
	{(const CHAR *) "volatile", (size_t) 8, kw_volatile, NIL_WORD},
	{NIL_CHAR, (size_t) 0, (TOKEN) 0, NIL_WORD}
    };

#ifdef EXTENSION
    static XWORD extension_keywords[] =
    {
	{(const CHAR *) "complex", (size_t) 7, kw_complex, NIL_WORD},
	{(const CHAR *) "imaginary", (size_t) 9, kw_imaginary, NIL_WORD},
	{(const CHAR *) "inline", (size_t) 6, kw_inline, NIL_WORD},
	{(const CHAR *) "restrict", (size_t) 8, kw_restrict, NIL_WORD},
	{NIL_CHAR, (size_t) 0, (TOKEN) 0, NIL_WORD}
    };

#endif /* EXTENSION */
    WPTR    wp;
    HASH    keyno;

    for (wp = &keywords[0]; wp->spelling != NIL_CHAR; wp++) {
	keyno = hash (wp->spelling, wp->length);
	wp->next = hashtable[keyno];
	hashtable[keyno] = wp;
    }
    if (!trad_option) {
	for (wp = &ansi_keywords[0]; wp->spelling != NIL_CHAR; wp++) {
	    keyno = hash (wp->spelling, wp->length);
	    wp->next = hashtable[keyno];
	    hashtable[keyno] = wp;
	}
    }
#ifdef EXTENSION
    if (extension_option) {
	for (wp = &extension_keywords[0]; wp->spelling != NIL_CHAR; wp++) {
	    keyno = hash (wp->spelling, wp->length);
	    wp->next = hashtable[keyno];
	    hashtable[keyno] = wp;
	}
    }
#endif /* EXTENSION */
#ifdef ASM
    if (asm_option) {
	VOIDCAST quick_insert ((const CHAR *) "asm", (size_t) 3, kw_asm);
    }
#ifdef FACIST
    else {
	VOIDCAST quick_insert ((const CHAR *) "asm", (size_t) 3, kw_id);
    }
#endif /* FACIST */
#endif /* ASM */

    /*
     *  Special names recognised by lower parts of the compiler in
     *  order to do special actions.
     */
    alloca_name = quick_insert ((const CHAR *) "alloca", (size_t) 6, tk_id);
    printf_name = quick_insert ((const CHAR *) "printf", (size_t) 6, tk_id);
    fprintf_name = quick_insert ((const CHAR *) "fprintf", (size_t) 7, tk_id);
    sprintf_name = quick_insert ((const CHAR *) "sprintf", (size_t) 7, tk_id);
    scanf_name = quick_insert ((const CHAR *) "scanf", (size_t) 5, tk_id);
    fscanf_name = quick_insert ((const CHAR *) "fscanf", (size_t) 6, tk_id);
    sscanf_name = quick_insert ((const CHAR *) "sscanf", (size_t) 6, tk_id);

    /*
     *  Names used for pre-processor directives
     */
    pp_pragma = quick_insert ((const CHAR *) "pragma", (size_t) 6, tk_id);
    pp_ident = quick_insert ((const CHAR *) "ident", (size_t) 5, tk_id);
    pp_line = quick_insert ((const CHAR *) "line", (size_t) 4, tk_id);
    pp_file = quick_insert ((const CHAR *) "file", (size_t) 4, tk_id);

#ifdef TOPSPEED
    /*
     *  TopSpeed keywords
     */
    if (topspeed_option) {
	VOIDCAST quick_insert ((const CHAR *) "cdecl", (size_t) 5, kw_cdecl);
    }
#endif /* TOPSPEED */

    /* set up attributes */
    setattributes ((const CHAR *) "01234567",
	      (Attribute) (A_PRINT | A_IDCHAR | A_HEX | A_DIGIT | A_OCTAL));
    setattributes ((const CHAR *) "89",
		   (Attribute) (A_PRINT | A_IDCHAR | A_HEX | A_DIGIT));
    setattributes ((const CHAR *) "ABCDEFabcdef",
		   (Attribute) (A_PRINT | A_IDCHAR | A_HEX));
    setattributes ((const CHAR *) "GHIJKLMNOPQRSTUVWXYZghijklmnopqrstuvwxyz_",
		   (Attribute) (A_PRINT | A_IDCHAR));
    setattributes ((const CHAR *) " \f\n\r\t\v", A_SPACE);

    total_errors = 0;

    /*
     *  Initialize the input buffer
     */
    buflen = (size_t) (BUFLEN + ID_LENGTH);
    bufstart = (CHAR *) malloc (buflen + (size_t) 1);
    bufcur = bufstart;		/* at start of input buffer */
    bufend = bufstart + buflen;	/* pointer beyond end of input buffer */
    bufmove = bufstart;		/* buffer won't be extended */

    bufstart[0] = (CHAR) '\n';
    bufstart[1] = END_OF_BUFFER;
    buflimit = bufstart + 1;

    strbuflen = (size_t) BUFLEN;
    strstart = (CHAR *) malloc (strbuflen);
}


/*
 *  new_line() - this routine processes newline characters.
 */
static void new_line P0 (void)
{
    *bufcur = (CHAR) '\0';
    nextch ();
    ++act_line;
    if (*bufcur == (CHAR) '#') {
	preprocessor_directive ();
    } else {
	act_linetxt = bufcur;
    }
}

/*
 *  skip_space() - this routine skips whitespace characters.
 */
static void skip_space P0 (void)
{
    for (;;) {
	while (is_space (*bufcur)) {
	    if (*bufcur == (CHAR) '\n') {
		new_line ();
	    } else {
		bufcur++;
	    }
	}
	if (bufcur < buflimit) {
	    break;
	}
	if (end_of_file) {
	    break;
	}
	symstart = bufcur;
	nextline ();
    }
}

/*
 *  preprocessor_directive() - handling of pre-processor directives.
 *
 *  The line starts with #, so it is assumed to be a pre-processor
 *  directive.
 *
 *  #<num>
 *  #line <num>
 *  #line <num> <filename>
 *       is treated as a line number re-synchronisation directive
 *       and analysed further.    For compatibility with the DECUS
 *       pre-processor, # which is followed by a number is treated as
 *       equivalent to the #line directive.
 *
 *       This directive is expected to be followed by the line number.  A
 *       filename can also optionally be present (if omitted it is assumed
 *       that the filename has not changed since the last #line directive).
 *
 *  #file
 *       the line is ignored
 *
 *  #ident
 *       the line is ignored
 *
 *  #pragma
 *       an appropriate message output and the line is ignored.
 *
 *  Any other directive is treated as an error, an appropriate error
 *  message is output and the line ignored.
 */
static void preprocessor_directive P0 (void)
{
    int     i;
    size_t  len;
    const CHAR *ptr;

    symstart = bufcur;
    nextch ();			/* consume # */
    skip_space ();
    switch (*bufcur) {
    default:
	symstart = bufcur;
	while (is_idch (*bufcur)) {
	    nextch ();
	}
	len = (size_t) (bufcur - symstart);
	ptr = found (symstart, len, FALSE);
	if (ptr == pp_pragma) {
	    /*
	     *   #pragma
	     */
	    while ((*bufcur == (CHAR) ' ') || (*bufcur == (CHAR) '\t')) {
		nextch ();	/* remove leading white space */
	    }
	    symstart = bufcur;
	    while (*bufcur) {
		if (*bufcur == (CHAR) '\n') {
		    *bufcur = (CHAR) '\0';
		    options ((char *) symstart, TRUE);
		    *bufcur = (CHAR) '\n';
		    break;
		}
		nextch ();
	    }
	    return;
	}
	if (ptr == pp_ident) {
	    /*
	     *  #ident
	     */
	    while (*bufcur) {
		if (*bufcur == (CHAR) '\n') {
		    break;
		}
		nextch ();
	    }
	    return;
	}
	if (ptr == pp_file) {
	    /*
	     *  #file
	     */
	    while (*bufcur) {
		if (*bufcur == (CHAR) '\n') {
		    break;
		}
		nextch ();
	    }
	    return;
	}
	if (ptr != pp_line) {
	    message (ERR_PREPROC);
	    return;
	}
	/*
	 *  #line line-number [filename]
	 */
	skip_space ();
	if (!is_digit (*bufcur)) {
	    message (ERR_PREPROC);
	    return;
	}
	/*FALLTHRU */
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
	/*
	 *  #line-number [ filename]
	 */
	for (i = 0, act_line = (LINE) 0; is_digit (*bufcur); nextch ()) {
	    i = radix36 (*bufcur);
	    act_line = ((LINE) 10 * act_line) + (LINE) i;
	}

	/*
	 *  scan file name
	 *
	 *  If present it is assumed to be surrounded by double quotes.
	 *  As a safety measure, end-of-line will also terminate the filename.
	 *
	 *  DECUS cpp suppresses the name if it has not changed
	 *  in this case, and then we keep the old name
	 */
	while (*bufcur == (CHAR) ' ')
	    nextch ();
	switch (*bufcur) {
	case '"':
	    nextch ();		/* step over the initial quote */
	    /*FALLTHRU */
	default:
	    for (symstart = bufcur; *bufcur; nextch ()) {
		if (*bufcur == (CHAR) '"' || *bufcur == (CHAR) '\n') {
		    break;
		}
	    }
	    len = (size_t) (bufcur - symstart);
	    /*
	     *  Allocate the name into the spelling table.
	     */
	    act_file = found (symstart, len, FALSE);

	    /*
	     *  Ignore the rest of the line
	     */
	    while (*bufcur != (CHAR) '\n' && *bufcur != END_OF_BUFFER)
		nextch ();
	    break;
	case '\n':
	    break;
	}
	act_line--;
	return;
    }
}

/*
 *  nextch() - basic get character routine.
 */
static void nextch P0 (void)
{
    bufcur++;
    if (*bufcur == END_OF_BUFFER) {
	nextline ();
    }
}


/*
 *  getid() - get an identifier.
 *
 *  Identifiers are any is_idch conglomerate that doesn't start
 *  with a numeric character. This set INCLUDES keywords.
 */
static void getid P0 (void)
{
    register size_t len;

    if (!trad_option && *bufcur == (CHAR) 'L') {
	/* Grrr!!  ANSI - wide character constants start with 'L' */
	nextch ();
	switch (*bufcur) {
	case '\'':
	    nextch ();
	    symstart = bufcur;
	    ival = (UVAL) getsch (FALSE);	/* get a string char */
	    if (*bufcur == (CHAR) '\'') {
		nextch ();
	    } else {
		message (ERR_CHARCONST);
	    }
	    lastst = tk_wconst;
	    return;
	case '\"':
	    len = (size_t) 0;
	    do {
		nextch ();
		for (;; ++len) {
		    int     j;

		    if ((j = getsch (TRUE)) == END_STRING) {
			break;
		    }
		    symstart[len] = (CHAR) j;
		}
		if (*bufcur == (CHAR) '\"') {
		    nextch ();
		} else {
		    message (ERR_STRINGCONST);
		}
		skip_space ();
	    } while (*bufcur == (CHAR) '\"');
	    /*
	     *  By looking up the string in the spelling table we ensure
	     *  that the same strings will return the same pointer ... thus
	     *  removing the necessity for comparing strings elsewhere in
	     *  the compiler.
	     */
	    lastsym = found (symstart, len, FALSE);
	    lastsymlen = len;
	    lastst = tk_wsconst;
	    return;
	default:
	    break;
	}
    }
    for (;;) {
	while (is_idch (*bufcur)) {
	    bufcur++;
	}
	if (bufcur < buflimit) {
	    break;
	}
	/*
	 *   The input buffer ran out ... get more input!
	 */
	nextline ();
    }
    len = (size_t) (bufcur - symstart);
    lastsym = found (symstart, len, TRUE);
}


/*
 *  getsch() - get a character in a quoted string.
 *
 *  This routine handles all of the escape mechanisms for characters
 *  in strings and character constants.
 *      is_string is FALSE, if a character constant is being scanned,
 *      is_string is TRUE, if a string constant is being scanned
 */
static int getsch P1 (BOOL, is_string)
{
    register int i, j;

    /*
     *  if we scan a string constant, stop if '"' is seen
     */
    switch (*bufcur) {
    case '\n':
	return END_STRING;
    case '"':
	if (is_string) {
	    return END_STRING;
	}
	/*FALLTHRU */
    case '\0':
	if (bufcur >= buflimit) {
	    /*
	     *  The input buffer ran out ... get more input!
	     */
	    nextline ();
	    return getsch (is_string);
	}
	/*FALLTHRU */
    default:
	i = (int) (*bufcur) & 0377;
	nextch ();
	/* signed characters lie in the range -128..127 */
	if (!uchar_option && (i >= 128)) {
	    i -= 256;
	}
	return i;
    case '\\':
	nextch ();		/* get an escaped character */
	switch (*bufcur) {
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	    i = 0;
	    for (j = 0; (j < 3) && is_octal (*bufcur); ++j) {
		i = 8 * i + radix36 (*bufcur);
		nextch ();
	    }
	    /*
	     * constraint: The value of an octal escape sequence shall be
	     *         in the range of representable values for the type
	     *         unsigned char for an integer character constant, or
	     *         an the unsigned type corresponding to wchar_t for a
	     *         wide character constant.
	     */
	    if (i > 256) {
		message (ERR_ESCRANGE);
	    }
	    i &= 0377;
	    /* signed characters lie in the range -128..127 */
	    if (!uchar_option && (i >= 128)) {
		i -= 256;
	    }
	    return i;
	case 'x':
	    nextch ();
	    /* get hexadecimal character */
	    i = 0;
	    for (j = 0; is_hex (*bufcur); j++) {
		i = 16 * i + radix36 (*bufcur);
		nextch ();
	    }
	    if (j == 0) {
		message (WARN_NOHEX);
	    }
	    /*
	     * constraint: The value of a hexadecimal escape sequence shall be
	     *         in the range of representable values for the type
	     *         unsigned char for an integer character constant, or
	     *         an the unsigned type corresponding to wchar_t for a
	     *         wide character constant.
	     */
	    if (i > 256) {
		message (ERR_ESCRANGE);
	    }
	    i &= 0377;
	    /* signed characters lie in the range -128..127 */
	    if (!uchar_option && (i >= 128)) {
		i -= 256;
	    }
	    return i;
	default:
	    i = (int) *bufcur;
	    nextch ();
	    switch (i) {
	    case '\n':
		return getsch (1);
	    case 'b':
		return '\b';
	    case 'f':
		return '\f';
	    case 'n':
		return '\n';
	    case 'r':
		return '\r';
	    case 't':
		return '\t';
#if 0
	    case 'a':
		return '\a';
	    case 'v':
		return '\v';
#else
	    case 'a':
		return '\007';
	    case 'v':
		return '\013';
#endif
	    default:
		message (WARN_ESCAPECH, i);
		/*FALLTHRU */
	    case '\\':
	    case '\'':
	    case '\"':
		return i;
	    }
	}
    }
}

/*
 *  radix36() - return the numeric value for a character
 *
 *  This function makes assumptions about the character codes.
 */
static int radix36 P1 (CHAR, c)
{
    if (c >= (CHAR) '0' && c <= (CHAR) '9') {
	return (int) c - '0';
    }
    if (c >= (CHAR) 'a' && c <= (CHAR) 'z') {
	return (int) c - 'a' + 10;
    }
    if (c >= (CHAR) 'A' && c <= (CHAR) 'Z') {
	return (int) c - 'A' + 10;
    }
    return -1;
}

/*
 *  test_int() - Test on long or unsigned constants.
 *
 *  The type of integer constants depends on its value, representation
 *  and suffixes:
 *
 *  unsuffixed decimals are of type int, long int, unsigned long int
 *  unsuffixed octals or hexadecimals are int, unsigned int, long int
 *  or unsigned long int
 *
 *  constants with suffix U are unsigned int or unsigned long int
 *  constants with suffix L are long int or unsigned long int
 *  constants with suffixes L and U are unsigned long int
 *   (X3.159-1989 3.1.3.2)
 *
 *   The variable rep is a bitmask describing the representation of
 *   the constant.
 */

#define	H	1		/* hex or octal (this is passed in from caller) */
#define	U	2		/* U suffix */
#define	L	4		/* L suffix */
#define	S	8		/* ints are short */

static void test_int P1 (BOOL, rep)
{
    static struct {
	UVAL    minval;
	int     repmask;
	TOKEN   type;
	const char *msg;
    } typetab[] = {
	{			/* U and L */
	    Ox0UL, U | L, tk_ulconst, (const char *) NULL
	},
	{			/* >31 bits, short int */
	    Ox80000000UL, S, tk_ulconst, "unsigned long"
	},
	{			/* U */
	    Ox0UL, U, tk_uconst, (const char *) NULL
	},
	{			/* L >31 bits */
	    Ox80000000UL, L, tk_ulconst, "unsigned long"
	},
	{			/* >31 bits */
	    Ox80000000UL, 0, tk_uconst, "unsigned"
	},
	{			/* U and >16 bits, short int */
	    Ox00010000UL, S | U, tk_ulconst, "unsigned long"
	},
	{			/* L */
	    Ox0UL, L, tk_lconst, (const char *) NULL
	},
	{			/* >16 bits, short int */
	    Ox00010000UL, S, tk_lconst, "long"
	},
	{			/* U */
	    Ox0UL, S | U, tk_uconst, (const char *) NULL
	},
	{			/* >15 bits, hex/octal, short int */
	    Ox8000UL, S | H, tk_uconst, "unsigned"
	},
	{			/* >15 bits, short int */
	    Ox8000UL, S, tk_lconst, "long"
	},
	{			/* int */
	    Ox0UL, 0, tk_iconst, (const char *) NULL
	}
    }, *typeptr;

    assert (rep <= 1);

    if (overflow) {
	message (ERR_REPRESENT);
    }
    if (tp_int->size == tp_short->size) {
	rep |= S;
    }
    /*
     *  first check suffixes.
     *  L, U, and combinations are allowed.
     */
    for (;;) {
	if (*bufcur == (CHAR) 'l' || *bufcur == (CHAR) 'L') {
	    if (rep & L) {
		break;
	    }
	    rep |= L;
	} else if (*bufcur == (CHAR) 'u' || *bufcur == (CHAR) 'U') {
	    if (rep & U) {
		break;
	    }
	    rep |= U;
	} else {
	    break;
	}
	nextch ();
    }

    /*
     *  now search table for first match
     */
    for (typeptr = typetab;; typeptr++) {
	if ((rep & typeptr->repmask) != typeptr->repmask) {
	    continue;
	}
	if (ival >= typeptr->minval) {
	    if (typeptr->msg) {
		message (WARN_CONSTANT, typeptr->msg);
	    }
	    break;
	}
    }
    lastst = typeptr->type;
}

/*
 *  getdecimal() - get integers
 *
 *  rval is computed simultaneously - this handles floating point
 *  constants whose integer part is greater than INT_MAX correctly,
 *  e. g. 10000000000000000000000.0
 */
static void getdecimal P0 (void)
{
    register UVAL i = (UVAL) 0;
    register int j;
    UVAL    old = (UVAL) 0;

#ifdef FLOAT_SUPPORT
#ifndef FLOAT_BOOTSTRAP
    register RVAL r = F_zero;

#endif /* FLOAT_BOOTSTRAP */
#endif /* FLOAT_SUPPORT */

    while (is_digit (*bufcur)) {
	j = radix36 (*bufcur);
	i = ((UVAL) 10 * i) + (UVAL) j;
	if (old > i) {
	    /* must have wrapped the constant ... too big */
	    overflow = TRUE;
	}
	old = i;
#ifdef FLOAT_SUPPORT
#ifndef FLOAT_BOOTSTRAP
	r = F_ten * r + (RVAL) j;
#endif /* FLOAT_BOOTSTRAP */
#endif /* FLOAT_SUPPORT */
	nextch ();
    }
    ival = i;
#ifdef FLOAT_SUPPORT
#ifndef FLOAT_BOOTSTRAP
    rval = r;
#endif /* FLOAT_BOOTSTRAP */
#endif /* FLOAT_SUPPORT */
}

/*
 *  getoctal() - get an octal number
 */
static void getoctal P0 (void)
{
    register UVAL i = (UVAL) 0;
    register int j;
    UVAL    old = (UVAL) 0;

    while (is_octal (*bufcur)) {
	j = radix36 (*bufcur);
	i = ((UVAL) 8 * i) + (UVAL) j;
	if (old > i) {
	    /* must have wrapped the constant ... too big */
	    overflow = TRUE;
	}
	old = i;
	nextch ();
    }
    ival = i;
}

/*
 *  gethex() - get a hexadecimal number.
 */
static void gethex P0 (void)
{
    register UVAL i = (UVAL) 0;
    register int j;
    UVAL    old = (UVAL) 0;

    if (!is_hex (*bufcur)) {
	static CHAR *temp = (CHAR *)" ";

	temp[0] = *bufcur;
	message (ERR_PUNCT, temp);
    }
    while (is_hex (*bufcur)) {
	j = radix36 (*bufcur);
	i = ((UVAL) 16 * i) + (UVAL) j;
	if (old > i) {
	    /* must have wrapped the constant ... too big */
	    overflow = TRUE;
	}
	old = i;
	nextch ();
    }
    ival = i;
}

/*
 *  getfrac() - get fraction part of a floating number.
 *
 *  Since 0.1 has no exact representation in a binary base,
 *  we divide by 10.0 instead of multiplying by 0.1
 *  (Thanks to Michael Mueller for this comment)
 *  Improved algorithym by doing a single division at the end
 *  - supplied by Tom Watson.
 */
static void getfrac P0 (void)
{
#ifdef FLOAT_SUPPORT
#ifndef FLOAT_BOOTSTRAP
    RVAL    frmul;
    RVAL    fraction;

    frmul = F_one;
    fraction = F_zero;
    while (is_digit (*bufcur)) {
	frmul *= F_ten;
	fraction *= F_ten;
	fraction += (RVAL) radix36 (*bufcur);
	nextch ();
    }
    rval += fraction / frmul;
    lastst = tk_rconst;
#endif /* FLOAT_BOOTSTRAP */
#endif /* FLOAT_SUPPORT */
}

/*
 *  getexp() - get exponent part of floating number.
 *
 *  This algorithm is primitive but useful.
 *  We have to limit the exponent range somehow since 'endless'
 *  loops can occur if 1E123456789 is to be converted.
 *  The biggest exponent range I am aware of (CRAY Y-MP) is
 *  2^4096, say 10^1200, so I limit the exponent range to 1500.
 *  Is it better to flag an error or to generate +/- inf or
 *  zero here?
 *
 *  The reason we divide by exmul, and not multiply by 1/exmul is that
 *  0.1 can NOT be represented as an exact quantity in a binary base.
 *  Thus, when you increase sizeof(double), you get into trouble.
 */
static void getexp P0 (void)
{
#ifdef FLOAT_SUPPORT
#ifndef FLOAT_BOOTSTRAP
    RVAL    expo;
    RVAL    multiplier;
    int     flag = FALSE;

    expo = rval;
    multiplier = F_ten;
    switch (*bufcur) {
    case '-':
	flag = TRUE;
	/*FALLTHRU */
    case '+':
	nextch ();
	break;
    default:
	break;
    }
    if (!is_digit (*bufcur)) {
	static CHAR temp[] = " ";

	temp[0] = *bufcur;
	message (ERR_PUNCT, temp);
    }
    getdecimal ();
    if (ival < (UVAL) 1500) {
	while (ival-- != (UVAL) 0) {
	    if (flag) {
		expo /= multiplier;
	    } else {
		expo *= multiplier;
	    }
	}
    } else {
	message (ERR_CONSTFLOAT);
    }
    rval = expo;
    lastst = tk_rconst;
#endif /* FLOAT_BOOTSTRAP */
#endif /* FLOAT_SUPPORT */
}


/*
 *  getnum() - get a number from input.
 *
 *  getnum handles all of the numeric input. it accepts decimal, octal,
 *  hexidecimal, and floating point numbers.
 */
static void getnum P0 (void)
{
    overflow = FALSE;
    lastst = tk_iconst;
    switch (*bufcur) {
    case '0':
	nextch ();
	switch (*bufcur) {
	case 'x':
	case 'X':
	    nextch ();
	    gethex ();
	    test_int (TRUE);
	    return;
	default:
	    getoctal ();
	    test_int (TRUE);
	    return;
	case '.':
	case 'e':
	case 'E':
	    break;
	}
	/*FALLTHRU */
    default:
	getdecimal ();
	switch (*bufcur) {
	case '.':
	    /* rval already set */
	    nextch ();
	    getfrac ();		/* add the fractional part */
	    /*FALLTHRU */
	default:
	    switch (*bufcur) {
	    case 'e':
	    case 'E':
		nextch ();
		getexp ();	/* get the exponent */
		/*FALLTHRU */
	    default:
		if (lastst == tk_rconst) {
		    switch (*bufcur) {
		    case 'f':
		    case 'F':
			lastst = tk_fconst;
			nextch ();
			break;
		    case 'l':
		    case 'L':
			lastst = tk_lrconst;
			nextch ();
			break;
		    default:
			break;
		    }
		} else {
		    /*
		     * look for l and u qualifiers
		     */
		    test_int (FALSE);
		}
		break;
	    }
	    break;
	}
    }
}

/*
 *  getsym() - get next symbol from input stream.
 *
 *  getsym is the basic lexical analyser.
 *  It builds basic tokens out of the characters on the input stream
 *  and sets the following global variables:
 *
 *  lastst:         type of last symbol read.
 *  lastsym:        last identifier/string read.
 *  ival:           last integer constant read.
 *  rval:           last real constant read.
 *
 *  getsym should be called for all your input needs...
 */
void getsym P0 (void)
{
    register size_t i;

    for (;;) {
      restart:
	symstart = bufcur;
	switch (*bufcur) {
	case '\n':
	    new_line ();
	    continue;
	case '\f':
	case '\r':
	case '\t':
	case '\v':
	case ' ':
	    bufcur++;
	    continue;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	    getnum ();
	    break;
	case 'A':
	case 'B':
	case 'C':
	case 'D':
	case 'E':
	case 'F':
	case 'G':
	case 'H':
	case 'I':
	case 'J':
	case 'K':
	case 'L':
	case 'M':
	case 'N':
	case 'O':
	case 'P':
	case 'Q':
	case 'R':
	case 'S':
	case 'T':
	case 'U':
	case 'V':
	case 'W':
	case 'X':
	case 'Y':
	case 'Z':
	case 'a':
	case 'b':
	case 'c':
	case 'd':
	case 'e':
	case 'f':
	case 'g':
	case 'h':
	case 'i':
	case 'j':
	case 'k':
	case 'l':
	case 'm':
	case 'n':
	case 'o':
	case 'p':
	case 'q':
	case 'r':
	case 's':
	case 't':
	case 'u':
	case 'v':
	case 'w':
	case 'x':
	case 'y':
	case 'z':
	case '_':
	    getid ();
	    break;
	case '+':
	    nextch ();
	    switch (*bufcur) {
	    case '+':
		lastst = tk_autoinc;
		bufcur++;
		break;
	    case '=':
		lastst = tk_asplus;
		bufcur++;
		break;
	    default:
		lastst = tk_plus;
		break;
	    }
	    break;
	case '-':
	    nextch ();
	    switch (*bufcur) {
	    case '-':
		lastst = tk_autodec;
		bufcur++;
		break;
	    case '=':
		lastst = tk_asminus;
		bufcur++;
		break;
	    case '>':
		lastst = tk_pointsto;
		bufcur++;
		break;
	    default:
		lastst = tk_minus;
		break;
	    }
	    break;
	case '*':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_astimes;
		bufcur++;
		break;
	    default:
		lastst = tk_star;
		break;
	    }
	    break;
	case '/':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_asdivide;
		bufcur++;
		break;
	    case '*':
		while (!end_of_file) {
		    symstart = bufcur;
		    nextch ();
		    if (*bufcur == (CHAR) '*') {
			nextch ();
			if (*bufcur == (CHAR) '/') {
			    bufcur++;
			    goto restart;
			}
		    }
		}
		break;
#ifdef EXTENSION
	    case '/':
		if (extension_option) {
		    while (!end_of_file) {
			if (*bufcur == (CHAR) '\n') {
			    new_line();
			    goto restart;
			}
			nextch ();
		    }
		    break;
		}
		/*FALLTHRU */
#endif /* EXTENSION */
	    default:
		lastst = tk_divide;
		break;
	    }
	    break;
	case '^':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_asuparrow;
		bufcur++;
		break;
	    default:
		lastst = tk_uparrow;
		break;
	    }
	    break;
	case ';':
	    lastst = tk_semicolon;
	    nextch ();
	    break;
	case ':':
	    nextch ();
	    lastst = tk_colon;
	    break;
	case '=':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_eq;
		bufcur++;
		break;
	    default:
		lastst = tk_assign;
		break;
	    }
	    break;
	case '>':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_geq;
		bufcur++;
		break;
	    case '>':
		nextch ();
		switch (*bufcur) {
		case '=':
		    lastst = tk_asrshift;
		    bufcur++;
		    break;
		default:
		    lastst = tk_rshift;
		    break;
		}
		break;
	    default:
		lastst = tk_gt;
		break;
	    }
	    break;
	case '<':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_leq;
		bufcur++;
		break;
#ifdef EXTENSION
	    case ':':
		lastst = tk_openbr;
		bufcur++;
		break;
	    case '%':
		lastst = tk_begin;
		bufcur++;
		break;
#endif /* EXTENSION */
	    case '<':
		nextch ();
		switch (*bufcur) {
		case '=':
		    lastst = tk_aslshift;
		    bufcur++;
		    break;
		default:
		    lastst = tk_lshift;
		    break;
		}
		break;
	    default:
		lastst = tk_lt;
		break;
	    }
	    break;
	case '\'':
	    nextch ();
	    ival = (UVAL) getsch (FALSE);	/* get a string char */
	    if (*bufcur == (CHAR) '\'') {
		bufcur++;
	    } else {
		message (ERR_CHARCONST);
	    }
	    lastst = tk_iconst;
	    break;
	case '\"':
	    i = (size_t) 0;
	    do {
		symstart = bufcur;
		nextch ();
		for (;; ++i) {
		    register int j;

		    if ((j = getsch (TRUE)) == END_STRING) {
			break;
		    }
		    if (i == strbuflen) {
			strbuflen += (size_t) BUFLEN;
			strstart = (CHAR *) realloc (strstart, strbuflen);
		    }
		    strstart[i] = (CHAR) j;
		}
		if (*bufcur == (CHAR) '\"') {
		    nextch ();
		} else {
		    message (ERR_STRINGCONST);
		}
		skip_space ();
	    } while (*bufcur == (CHAR) '\"' && !trad_option);

	    /*
	     *  By looking up the string in the spelling table we ensure
	     *  that the same strings will return the same pointer ... thus
	     *  removing the necessity for comparing strings elsewhere in
	     *  the compiler.
	     */
	    lastsym = found (strstart, i, FALSE);
	    lastsymlen = i;
	    lastst = tk_sconst;
	    break;
	case '!':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_neq;
		bufcur++;
		break;
	    default:
		lastst = tk_not;
		break;
	    }
	    break;
	case '%':
	    nextch ();
	    switch (*bufcur) {
	    case '=':
		lastst = tk_asmod;
		bufcur++;
		break;
#ifdef EXTENSION
	    case '>':
		lastst = tk_end;
		bufcur++;
		break;
#endif /* EXTENSION */
	    default:
		lastst = tk_mod;
		break;
	    }
	    break;
	case '~':
	    nextch ();
	    switch (*bufcur) {
#ifdef EXTENSION
	    case '>':
		lastst = tk_closebr;
		bufcur++;
		break;
#endif /* EXTENSION */
	    default:
		lastst = tk_compl;
		break;
	    }
	    break;
	case '.':
	    nextch ();
	    switch (*bufcur) {
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
#ifdef FLOAT_SUPPORT
#ifndef FLOAT_BOOTSTRAP
		rval = F_zero;
#endif /* FLOAT_BOOTSTRAP */
#endif /* FLOAT_SUPPORT */
		getfrac ();
		switch (*bufcur) {
		case 'e':
		case 'E':
		    nextch ();
		    getexp ();
		    break;
		default:
		    break;
		}
		if (lastst == tk_rconst) {
		    switch (*bufcur) {
		    case 'f':
		    case 'F':
			lastst = tk_fconst;
			bufcur++;
			break;
		    case 'l':
		    case 'L':
			lastst = tk_lrconst;
			bufcur++;
			break;
		    default:
			break;
		    }
		}
		break;
	    case '.':
		nextch ();
		switch (*bufcur) {
		case '.':
		    lastst = tk_ellipsis;
		    bufcur++;
		    break;
		default:
		    message (ERR_PUNCT, "..");
		    continue;
		}
		break;
	    default:
		lastst = tk_dot;
		break;
	    }
	    break;
	case ',':
	    lastst = tk_comma;
	    nextch ();
	    break;
	case '&':
	    nextch ();
	    switch (*bufcur) {
	    case '&':
		lastst = tk_land;
		bufcur++;
		break;
	    case '=':
		lastst = tk_asand;
		bufcur++;
		break;
	    default:
		lastst = tk_and;
		break;
	    }
	    break;
	case '|':
	    nextch ();
	    switch (*bufcur) {
	    case '|':
		lastst = tk_lor;
		bufcur++;
		break;
	    case '=':
		lastst = tk_asor;
		bufcur++;
		break;
	    default:
		lastst = tk_or;
		break;
	    }
	    break;
	case '(':
	    lastst = tk_openpa;
	    nextch ();
	    break;
	case ')':
	    lastst = tk_closepa;
	    nextch ();
	    break;
	case '[':
	    lastst = tk_openbr;
	    nextch ();
	    break;
	case ']':
	    lastst = tk_closebr;
	    nextch ();
	    break;
	case '{':
	    lastst = tk_begin;
	    nextch ();
	    break;
	case '}':
	    lastst = tk_end;
	    nextch ();
	    break;
	case '?':
	    lastst = tk_hook;
	    nextch ();
	    break;
	case '\032':
	    /*
	     *	Some DOS editors leave a Control-Z character at the
	     *	end of the file.   Just to be nice we will ignore
	     *	this invalid character provided it really is the last
	     *	character!
	     */
	    if (end_of_file && (bufcur[1] == '\0')) {
		lastst = tk_eof;
		break;
	    }
	    message ((MSGNUM) (is_print (*bufcur) ? ERR_ILLCHAR : ERR_ILLXCHAR),
		     (int) *bufcur);
	    bufcur++;
	    continue;		/* get a real token */
	case '\0':
	    /*
	     *  There is always a zero byte at the end of the input buffer
	     *  so we have either just read a zero byte or have just
	     *  fallen off the end of the input buffer!
	     */
	    if (end_of_file) {
		lastst = tk_eof;
		break;
	    }
	    if (bufcur >= buflimit) {
		nextline ();
		continue;
	    }
	    /*FALLTHRU */
	default:
	    message ((MSGNUM) (is_print (*bufcur) ? ERR_ILLCHAR : ERR_ILLXCHAR),
		     (int) *bufcur);
	    bufcur++;
	    continue;		/* get a real token */
	}
	if (error_resync) {
	    error_resync--;
	}
	errorloop = FALSE;
	return;
    }
}

/*
 *  needpunc() - checks current token
 *
 *  Checks that the current token is the token specified by the
 *  parameter 'p'.  If it is the expected token then it moves onto
 *  the next token, otherwise it outputs an error message.
 */
void needpunc P1 (TOKEN, p)
{
    if (lastst == p) {
	getsym ();
    } else {
	if (lastst != tk_eof) {
	    CHAR    ch = *bufcur;

	    *bufcur = (CHAR) '\0';
	    message (ERR_PUNCT, symstart);
	    *bufcur = ch;
	} else {
	    message (ERR_EOF);
	    exit (EXIT_FAILURE);
	}
    }
}

/*
 *  is_label() - is current identifier a label
 *
 *  The is_label() routine "peeks" ahead in the input buffer to see
 *  if the identifier is followed by the ":" token.
 */
BOOL is_label P1 (TOKEN, p)
{
    if (p != tk_id) {
	return FALSE;
    }
    skip_space ();
    return (*bufcur == (CHAR) ':');
}
